﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.ComponentModel.Composition;
using System.ComponentModel.Composition.Hosting;
using System.Diagnostics;
using System.Runtime.InteropServices;

namespace OGSNET
{
    using OGSNET.Plugin;
    using OGSNET.Browser;

    public partial class MainForm : Form
    {
        private Timer Timer; /**< プラグインのタイムアウト処理をするためのタイマー */
        private PluginStarter PluginStarter = null; /**< プラグインの開始処理をするオブジェクト */
        private WebBrowser Browser = null; /**< 処理するWebBrowser */

        private const int IDM_VERSION       = 1000; /**< バージョン情報のメニューを表す識別子 */
        private const int IDM_OFFICIAL_PAGE = 1001; /**< 公式サイトを開くメニューを表す識別子 */

        /** \brief 初期化処理 */
        public MainForm()
        {
            // GUIの初期化
            InitializeComponent();

            // システムメニューの追加
            this.AddSystemMenu();

            // タイマーの初期化
            this.InitTimer();

            // アカウントデータの読み込み
            this.LoadAccountData();

            // プラグインの読み込み
            this.LoadPlugin();

            // ゲーム一覧を更新
            this.UpdateGameList();
            
            // イベントを追加する
            this.AddEvent();

            // プラグインのデバッグ
            this.PluginDebug();
        }

        /** \brief タイマーの初期化 */
        private void InitTimer()
        {
            this.Timer = new Timer();
            this.Timer.Interval = Properties.Settings.Default.PluginTimeout;
            this.Timer.Tick += new EventHandler(Timer_Tick);
        }

        /** \brief タイマーを起動してから一定時間経った場合の処理 */
        private void Timer_Tick(object sender, EventArgs e)
        {
            this.PluginTimeout();  
        }

        /** \biref プラグインのタイムアウト処理 */
        private void PluginTimeout()
        {
            this.Timer.Stop();

            // 解放処理
            if (this.PluginStarter != null)
            {
                this.PluginStarter.Dispose();
                this.PluginStarter = null;
            }

            // ブラウザをコントロールから削除
            if (this.Browser != null)
            {
                this.Controls.Remove(this.Browser);
                this.Browser = null;
            }
        }

        /** \brief プラグインを読み込む */
        private void LoadPlugin()
        {
            this.PluginControl = new PluginControl("plugins");
            this.OperationNameTable = new Dictionary<string, List<string>>();

            var plugins = this.PluginControl.GetPlugins();

            foreach (var accounts in this.AccountData)
            {
                var plugin = PluginControl.GetPlugin(plugins, accounts.Key);
                string operationName = null;

                // プラグインが存在しない
                if (plugin == null)
                {
                    continue;
                }

                // 運営プラグイン
                if (plugin.Value is IOperationPlugin)
                {
                    var operationPlugin = plugin.Value as IOperationPlugin;
                    operationName = operationPlugin.GetOperationInfo().OperationName;
                }

                // ゲームプラグイン
                else if(plugin.Value is IGamePlugin){
                    var gamePlugin = plugin.Value as IGamePlugin;
                    operationName = gamePlugin.GetGameInfo().OperationName;
                }

                if (!this.OperationNameTable.ContainsKey(operationName))
                {
                    this.OperationNameTable[operationName] = new List<string>();
                }

                this.OperationNameTable[operationName].Add(plugin.Metadata.PluginId);
            }
        }

        // アカウントを読み込む
        private void LoadAccountData()
        {
            var path = AccountUtility.GetSaveFolderPath();

            this.AccountDataControl =
                new AccountDataControl(path + @"\Account.dat", AccountUtility.GetCurrentUserSid());

            this.AccountDataControl.Load();

            this.AccountData = new Dictionary<string, List<AccountData>>();

            foreach (var account in this.AccountDataControl.Accounts)
            {
                if (!this.AccountData.ContainsKey(account.PluginId))
                {
                    this.AccountData[account.PluginId] = new List<AccountData>();
                }

                this.AccountData[account.PluginId].Add(account);
            }

        }

        /** \brief ゲーム一覧を更新 */
        private void UpdateGameList()
        {
            // 一覧をクリア
            this.cmbOperation.Items.Clear();
            this.cmbGame.Items.Clear();
            this.cmbAccount.Items.Clear();

            // 一覧を一時保存
            var operationNames = new List<string>();

            // 運営の名前に対してループ
            // Key: 運営の名前, Value: PluginId
            foreach (var operation in this.OperationNameTable)
            {
                bool findGame = false; // ゲームプラグインが存在するかどうか

                // その運営の名前に対するゲームが存在するか調べる
                foreach (var pluginId in operation.Value)
                {
                    if (this.PluginControl.GetPlugin<IGamePlugin>(pluginId) != null)
                    {
                        findGame = true;
                        break;
                    }
                }

                // ゲームが見つかったら、一時的な一覧へ追加
                if (findGame)
                {
                    operationNames.Add(operation.Key);
                }
            }
           
            // 運営名をソート
            operationNames.Sort();

            // 運営名を追加
            foreach (var operationName in operationNames)
            {
                this.cmbOperation.Items.Add(operationName);
            }

            // 運営名が一つ以上あれば、選択させる
            if (this.cmbOperation.Items.Count > 0)
            {
                this.cmbOperation.SelectedIndex = 0;
                this.ChangeOperation();
            }
        }

        /** \brief 運営が変更された場合の処理 */
        private void ChangeOperation()
        {
            // 一覧をクリア
            this.cmbGame.Items.Clear();
            this.cmbAccount.Items.Clear();
            
            // 現在選択中の運営名
            var name = this.cmbOperation.Text;
            var plugins = this.PluginControl.GetPlugins<IGamePlugin>();

            // ゲーム名を一時的に保管するリスト
            var gameNames = new List<StringPair>();

            // その名前の運営のゲームに対してループ
            foreach(var pluginId in this.OperationNameTable[name]){
                var plugin = PluginControl.GetPlugin(plugins, pluginId);

                // プラグインが存在する
                if (plugin != null)
                {
                    var pair = new StringPair(pluginId, plugin.Value.GetGameInfo().GameName);
                    gameNames.Add(pair); // 一時的に保存
                }
            }

            // 一時的に保存したゲーム名をソート
            gameNames.Sort();

            // 一覧を追加
            foreach (var gameName in gameNames)
            {
                this.cmbGame.Items.Add(gameName);
            }

            // 項目があれば、セレクトする
            if (this.cmbGame.Items.Count > 0)
            {
                this.cmbGame.SelectedIndex = 0;
                this.ChangeGame();
            }
        }

        /** \brief ゲームが変更されたときの処理 */
        private void ChangeGame()
        {
            // 一覧をクリア
            this.cmbAccount.Items.Clear();

            // 現在選択中のゲームを取得
            var gameInfo = (StringPair)this.cmbGame.SelectedItem;

            // 一時的に保存するアカウントの情報
            var accountTemp = new List<AccountData>();

            // ゲームIDから検索
            foreach (var account in this.AccountData[gameInfo.First])
            {
                if (account.IsEnable) // アカウントが有効
                {
                    accountTemp.Add(account);
                }
            }

            // 現在のゲームのプラグイン
            var plugin = this.PluginControl.GetPlugin<IGamePlugin>(gameInfo.First);

            // 現在のゲームの運営ID
            var operationId = plugin.Value.GetGameInfo().OperationId;

            // 運営IDが有効、かつアカウント一覧に運営プラグインが存在する場合
            if (operationId != null && this.AccountData.ContainsKey(operationId))
            {
                foreach (var account in this.AccountData[operationId])
                {
                    // アカウントが有効、かつ現在アカウントが存在しない場合
                    if (account.IsEnable && !this.AccountData[gameInfo.First].Contains(account))
                    {
                        AccountData accountNew = account;
                        accountNew.PluginId = gameInfo.First; // PluginId を運営からゲームプラグインのものへ変更

                        accountTemp.Add(accountNew);
                    }
                }
            }

            // ソート
            accountTemp.Sort();

            // 一時的に保存したアカウントを実際に追加
            foreach (var account in accountTemp)
            {
                this.cmbAccount.Items.Add(account);
            }

            // 項目が存在すれば、セレクト
            if (this.cmbAccount.Items.Count > 0)
            {
                this.cmbAccount.SelectedIndex = 0;
            }
        }

        /** \brief 選択されているゲームを開始する */
        private void GameStartSelected()
        {
            if (this.cmbAccount.Items.Count == 0) return;

            var account = (AccountData)this.cmbAccount.SelectedItem;

            this.GameStart(account);
        }

        /** \brief ゲームを開始する */
        private void GameStart(AccountData account, bool isDebugMode = false)
        {
            var plugin = this.PluginControl.GetPlugin(account.PluginId);

            if (plugin == null)
            {
                MessageBox.Show("プラグインが見つかりません。");
                Debug.WriteLine(account.PluginId);
                return;
            }

            // 処理する WebBrowser
            var browser = new WebBrowserEx();
            this.Browser = browser;
            browser.Visible = false;

            PluginStartInfo psi;

            psi.Account = (Account)account;
            psi.Browser = browser;
            psi.GetPlugin = new GetPlugin(this.GetPlugin);
            psi.StartCallback = new StartCallback(this.StartCallback);

            if (isDebugMode) // デバッグモード
            {
                browser.Visible = true;
                this.Controls.Clear();

                psi.Browser.Width  = 1024;
                psi.Browser.Height = 768;

                this.Width  = 1024;
                this.Height = 768;
            }

            // コントロールを追加
            this.Controls.Add(browser);

            // もしタイマーが実行中なら解放処理を行う
            if (this.Timer.Enabled)
            {
                this.PluginTimeout();
            }

            // タイマーを開始
            this.Timer.Start();

            // プラグインを起動
            this.PluginStarter = new PluginStarter(plugin.Value, psi);
        }

        /**
         * \brief プラグインデバッグモードを開始する
         * 
         * 画面にブラウザを表示し、他のコントロールは削除される
         */
        private void PluginDebug()
        {
            var args = CommandLineOption.GetInstance();

            if (!args.PluginDebug) return;
            if (args.PluginId == null || args.UserName == null || args.Password == null) return;

            var account = new AccountData();

            account.PluginId = args.PluginId;
            account.UserName = args.UserName;
            account.Password = args.Password;

            this.GameStart(account, true);
        }

        /** brief プラグインからのコールバックメソッド */
        private void StartCallback(CallbackStatus status, string message)
        {
            this.AddLog(CallbackStatusToString(status) + " : " + message + "\r\n");
        }

        /** \brief プラグインを取得するメソッド */
        private IPlugin GetPlugin(string pluginId){
            return this.PluginControl.GetPlugin(pluginId).Value;
        }

        /** \brief ログにテキストを追加する */
        private void AddLog(string message)
        {
            this.txtLog.AppendText(message);
            Debug.Write(message);
        }

        /** \brief CallbackStatus を文字列に直す */
        private string CallbackStatusToString(CallbackStatus status){
            switch(status){
                case CallbackStatus.Notice:
                case CallbackStatus.Finish:
                    return "メッセージ";
                case CallbackStatus.Warning:
                    return "警告";
                case CallbackStatus.Error:
                    return "エラー";
                default:
                    return "";
            }
        }

        /** \brief イベントを追加する */
        private void AddEvent(){
            this.cmbGame.SelectedIndexChanged += new EventHandler(this.cmbGame_SelectedIndexChanged);
            this.cmbOperation.SelectedIndexChanged += new EventHandler(this.cmbOperation_SelectedIndexChanged);
        }

        /** \brief イベントを削除する */
        private void RemoveEvent()
        {
            this.cmbGame.SelectedIndexChanged -= new EventHandler(this.cmbGame_SelectedIndexChanged);
            this.cmbOperation.SelectedIndexChanged -= new EventHandler(this.cmbOperation_SelectedIndexChanged);
        }

        /** \brief プラグインのコントロールオブジェクト */
        public PluginControl PluginControl
        {
            get;
            private set;
        }

        /** \brief アカウントデータのコントロールオブジェクト */
        public AccountDataControl AccountDataControl
        {
            get;
            private set;
        }

        /** \brief アカウントデータの連想配列
         * PluginId をキーとする
         */
        private Dictionary<string, List<AccountData>> AccountData
        {
            get;
            set;
        }

        /** \brief 運営の名前とプラグインのIDの対応表 */
        private Dictionary<string, List<string>> OperationNameTable
        {
            get;
            set;
        }

        /** \brief 運営が変更されたときの処理 */
        private void cmbOperation_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.ChangeOperation();
        }

        /** \brief ゲームが変更されたときの処理 */
        private void cmbGame_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.ChangeGame();
        }

        /** \brief ゲームを開始する */
        private void btnStart_Click(object sender, EventArgs e)
        {
            this.GameStartSelected();
        }

        /** \brief 設定画面を表示する */
        private void btnSetting_Click(object sender, EventArgs e)
        {
            using (Form setting = new SettingForm(this.PluginControl, this.AccountDataControl))
            {
                setting.ShowDialog();

                this.AccountDataControl.Save();

                // イベントを削除
                this.RemoveEvent();

                // アカウントデータの読み込み
                this.LoadAccountData();

                // プラグインの読み込み
                this.LoadPlugin();

                // ゲーム一覧を更新
                this.UpdateGameList();

                // イベントを追加
                this.AddEvent();
            }
        }

        /** \brief システムメニューを追加する */
        private void AddSystemMenu()
        {
            var hMenu = Win32API.GetSystemMenu(this.Handle, false);
            var count  = Win32API.GetMenuItemCount(hMenu);

            // セパレーターの追加
            Win32API.InsertMenu(hMenu, count++, Win32API.MF_BYPOSITION | Win32API.MF_SEPARATOR, 0, string.Empty);

            // 項目を追加
            Win32API.InsertMenu(hMenu, count++, Win32API.MF_BYPOSITION, MainForm.IDM_VERSION,       "バージョン情報 (&V)");
            Win32API.InsertMenu(hMenu, count++, Win32API.MF_BYPOSITION, MainForm.IDM_OFFICIAL_PAGE, "公式サイトを開く (&O)");
        }

        /** \brief メッセージの処理 */
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // システムメニューの処理
                case Win32API.WM_SYSCOMMAND:
                    switch (unchecked(m.WParam.ToInt32()))
                    {
                        // バージョン情報
                        case MainForm.IDM_VERSION:
                            MessageBox.Show(
                                Application.ProductName + " v" + Application.ProductVersion.ToString(),
                                "バージョン情報", MessageBoxButtons.OK, MessageBoxIcon.Information);
                            break;
                        
                        // 公式サイトを開く
                        case MainForm.IDM_OFFICIAL_PAGE:
                            try
                            {
                                Process.Start(Properties.Resources.OfficialPageUrl);
                            }
                            catch (Win32Exception) { }
                            break;
                    }
                    break;
            }
    
            base.WndProc(ref m);
        }
    }
}
